<?php
// webhook.php - ورودی اصلی وبهوک ربات

require __DIR__ . '/bootstrap.php';

$update = json_decode(file_get_contents('php://input'), true);

if (!$update) {
    exit('no update');
}

// ---------------- MESSAGE ----------------
if (isset($update['message'])) {
    $message = $update['message'];
    $chatId  = $message['chat']['id'];
    $from    = $message['from'];
    $text    = $message['text'] ?? null;

    $user   = getUser($pdo, $from);
    $userId = $user['id'];

    // اگر عکس/فایل است (رسید کارت به کارت)
    if (isset($message['photo']) || isset($message['document'])) {
        handleReceiptUpload($message, $user, $pdo, $telegram);
        exit;
    }

    // دستورات پایه
    if ($text === '/start') {
        $telegram->sendMessage(
            $chatId,
            "سلام {$user['first_name']} 👋\nبه ربات فروش خدمات تلگرام خوش آمدید.",
            Keyboard::userMain()
        );
        clearState($pdo, $userId);
        exit;
    }

    if ($text === '/admin') {
        if (!isAdmin($pdo, $from['id'])) {
            $telegram->sendMessage($chatId, "❌ شما ادمین نیستید.");
            exit;
        }
        $telegram->sendMessage($chatId, "📊 پنل مدیریت فعال شد.", Keyboard::adminMain());
        clearState($pdo, $userId);
        exit;
    }

    // اگر ادمین است، اول منوی ادمین چک شود
    if (isAdmin($pdo, $from['id']) && $text) {
        if (handleAdminText($text, $user, $pdo, $telegram)) {
            exit;
        }
    }

    // پردازش منوی کاربر
    handleUserText($text, $user, $pdo, $telegram);
    exit;
}

// ---------------- CALLBACK ----------------
if (isset($update['callback_query'])) {
    $cb     = $update['callback_query'];
    $data   = $cb['data'];
    $chatId = $cb['message']['chat']['id'];
    $msgId  = $cb['message']['message_id'];
    $from   = $cb['from'];

    $user   = getUser($pdo, $from);
    $userId = $user['id'];

    handleCallback($data, $cb['id'], $chatId, $msgId, $user, $pdo, $telegram, $callino);
    exit;
}

// =======================================================
// ================= USER HANDLERS =======================
// =======================================================

function handleUserText($text, $user, PDO $pdo, Telegram $telegram) {
    $chatId = $user['tg_id'];
    [$state, $data] = getState($pdo, $user['id']);

    switch ($text) {
        case '🎫 خرید پریمیوم':
            listServicesByType('premium', $chatId, $pdo, $telegram);
            return;

        case '⭐ خرید استارز':
            listServicesByType('stars', $chatId, $pdo, $telegram);
            return;

        case '📱 شماره مجازی':
            listServicesByType('number', $chatId, $pdo, $telegram);
            return;

        case '💼 پروفایل من':
            showProfile($user, $pdo, $telegram);
            return;

        case '📜 سفارشات من':
            listUserOrders($user, $pdo, $telegram);
            return;

        case 'ℹ️ راهنما / پشتیبانی':
            $telegram->sendMessage($chatId, "برای پشتیبانی به آیدی زیر پیام دهید:\n@YourSupport");
            return;

        default:
            // اگر state خاصی داشتیم (مثلاً وارد کردن یوزرنیم یا شماره)
            if ($state === 'enter_target') {
                $data['target'] = $text;
                createOrderForUser($user, $data, $pdo, $telegram);
                clearState($pdo, $user['id']);
                return;
            }
            if ($state === 'enter_quantity') {
                if (!ctype_digit($text) || (int)$text <= 0) {
                    $telegram->sendMessage($chatId, "❗ لطفاً یک عدد معتبر بفرستید.");
                    return;
                }
                $data['quantity'] = (int)$text;
                setState($pdo, $user['id'], 'enter_target', $data);
                $telegram->sendMessage($chatId, "حالا یوزرنیم (یا شماره) مورد نظر را ارسال کنید:");
                return;
            }

            $telegram->sendMessage($chatId, "از منوی زیر استفاده کنید:", Keyboard::userMain());
    }
}

function listServicesByType($type, $chatId, PDO $pdo, Telegram $telegram) {
    $stmt = $pdo->prepare("SELECT * FROM services WHERE type = ? AND active = 1");
    $stmt->execute([$type]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!$rows) {
        $telegram->sendMessage($chatId, "فعلاً خدمتی برای این بخش تعریف نشده است.");
        return;
    }

    $inline = [];
    foreach ($rows as $row) {
        $final = calcFinalPrice($row['base_price'], $row['profit_fixed'], $row['profit_percent']);
        $title = $row['title_fa'] . " - " . number_format($final) . " تومان";
        $inline[] = [
            ['text' => $title, 'callback_data' => "buy_service:{$row['id']}"]
        ];
    }
    $inline[] = [
        ['text' => 'انصراف ⬅️', 'callback_data' => 'cancel']
    ];

    $telegram->sendMessage($chatId, "یکی از گزینه‌ها را انتخاب کنید:", [
        'inline_keyboard' => $inline
    ]);
}

function calcFinalPrice($base, $fixed, $percent) {
    $amount = $base + $fixed;
    if ($percent > 0) {
        $amount += intval($base * $percent / 100);
    }
    return $amount;
}

function showProfile($user, PDO $pdo, Telegram $telegram) {
    $chatId = $user['tg_id'];
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM orders WHERE user_id = ?");
    $stmt->execute([$user['id']]);
    $count = $stmt->fetchColumn();

    $txt = "💼 پروفایل شما:\n"
         . "نام: {$user['first_name']}\n"
         . "یوزرنیم: @" . ($user['username'] ?? '-') . "\n"
         . "موجودی: " . number_format($user['balance']) . " تومان\n"
         . "تعداد سفارش‌ها: {$count}";
    $telegram->sendMessage($chatId, $txt);
}

function listUserOrders($user, PDO $pdo, Telegram $telegram) {
    $chatId = $user['tg_id'];
    $stmt = $pdo->prepare("
        SELECT o.id, o.status, o.final_price, s.title_fa
        FROM orders o
        JOIN services s ON s.id = o.service_id
        WHERE o.user_id = ?
        ORDER BY o.id DESC
        LIMIT 10
    ");
    $stmt->execute([$user['id']]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!$rows) {
        $telegram->sendMessage($chatId, "هنوز سفارشی ثبت نکرده‌اید.");
        return;
    }

    $lines = [];
    foreach ($rows as $r) {
        $lines[] = "#{$r['id']} | {$r['title_fa']} | " . number_format($r['final_price']) . " | وضعیت: {$r['status']}";
    }
    $telegram->sendMessage($chatId, implode(\"\\n\", $lines));
}

// =======================================================
// ================= CALLBACK HANDLER ====================
// =======================================================

function handleCallback($data, $cbId, $chatId, $msgId, $user, PDO $pdo, Telegram $telegram, CallinoApi $callino) {
    $telegram->answerCallbackQuery($cbId);

    if ($data === 'cancel') {
        $telegram->sendMessage($chatId, "عملیات لغو شد.", Keyboard::userMain());
        clearState($pdo, $user['id']);
        return;
    }

    if (strpos($data, 'buy_service:') === 0) {
        $serviceId = (int) substr($data, strlen('buy_service:'));
        $stmt = $pdo->prepare("SELECT * FROM services WHERE id = ?");
        $stmt->execute([$serviceId]);
        $service = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$service) {
            $telegram->sendMessage($chatId, "خدمت یافت نشد.");
            return;
        }

        $final = calcFinalPrice($service['base_price'], $service['profit_fixed'], $service['profit_percent']);

        $txt = "شما انتخاب کردید:\n"
             . "خدمت: {$service['title_fa']}\n"
             . "مبلغ پرداخت: " . number_format($final) . " تومان\n\n";

        // اگر نوع استارز است، اول تعداد را بپرس
        $data = [
            'service_id'  => $service['id'],
            'final_price' => $final,
            'type'        => $service['type'],
        ];

        if ($service['type'] === 'stars') {
            $txt .= "لطفاً تعداد استارز مورد نیاز را ارسال کنید:";
            setState($pdo, $user['id'], 'enter_quantity', $data);
        } else {
            $txt .= "لطفاً یوزرنیم (یا شماره) مورد نظر را ارسال کنید:";
            setState($pdo, $user['id'], 'enter_target', $data);
        }

        $telegram->sendMessage($chatId, $txt);
        return;
    }
}

// =======================================================
// ============== ORDER & PAYMENT LOGIC ==================
// =======================================================

function createOrderForUser($user, array $data, PDO $pdo, Telegram $telegram) {
    $chatId = $user['tg_id'];
    $serviceId  = $data['service_id'];
    $finalPrice = $data['final_price'];
    $target     = $data['target'];
    $quantity   = isset($data['quantity']) ? (int)$data['quantity'] : 1;

    $stmt = $pdo->prepare("INSERT INTO orders (user_id, service_id, target, final_price, quantity, status) VALUES (?,?,?,?,?, 'waiting_receipt')");
    $stmt->execute([$user['id'], $serviceId, $target, $finalPrice, $quantity]);
    $orderId = $pdo->lastInsertId();

    $stmt = $pdo->prepare("INSERT INTO payments (user_id, order_id, amount) VALUES (?,?,?)");
    $stmt->execute([$user['id'], $orderId, $finalPrice]);

    $txt = "سفارش #{$orderId} ثبت شد ✅\n"
         . "مبلغ: " . number_format($finalPrice) . " تومان\n"
         . "هدف: {$target}\n"
         . "تعداد: {$quantity}\n\n"
         . CARD_INFO_FA . "\n\n"
         . "پس از واریز، رسید (عکس یا فایل) را در همین چت ارسال کنید.";

    $telegram->sendMessage($chatId, $txt, Keyboard::userMain());
}

// رسید کارت به کارت از کاربر
function handleReceiptUpload(array $message, $user, PDO $pdo, Telegram $telegram) {
    $chatId = $user['tg_id'];

    $stmt = $pdo->prepare("
        SELECT id, order_id FROM payments
        WHERE user_id = ? AND status = 'pending'
        ORDER BY id DESC LIMIT 1
    ");
    $stmt->execute([$user['id']]);
    $pay = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$pay) {
        $telegram->sendMessage($chatId, "هیچ پرداخت در انتظار تأیید یافت نشد.");
        return;
    }

    if (isset($message['photo'])) {
        $photos = $message['photo'];
        $fileId = end($photos)['file_id'];
    } else {
        $fileId = $message['document']['file_id'];
    }

    $stmt = $pdo->prepare("UPDATE payments SET receipt_file_id = ?, status = 'pending' WHERE id = ?");
    $stmt->execute([$fileId, $pay['id']]);

    // ارسال برای ادمین اصلی
    $caption = "رسید جدید برای پرداخت #{$pay['id']} (سفارش #{$pay['order_id']}) از کاربر {$user['tg_id']}";
    $telegram->request('sendPhoto', [
        'chat_id' => OWNER_ID,
        'photo'   => $fileId,
        'caption' => $caption . \"\\n\\nبرای تأیید: /approve_{$pay['id']}\\nبرای رد: /reject_{$pay['id']}\"
    ]);

    $telegram->sendMessage($chatId, "✅ رسید شما ثبت شد.\nپس از بررسی ادمین نتیجه اعلام می‌شود.");
}

// =======================================================
// ================= ADMIN HANDLERS ======================
// =======================================================

function handleAdminText($text, $user, PDO $pdo, Telegram $telegram) {
    $chatId = $user['tg_id'];

    // دکمه بازگشت
    if ($text === 'بازگشت به منوی کاربر ⬅️') {
        $telegram->sendMessage($chatId, "بازگشت به منوی کاربر.", Keyboard::userMain());
        return true;
    }

    // آمار ربات
    if ($text === 'آمار ربات 📊') {
        $users  = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
        $orders = $pdo->query("SELECT COUNT(*) FROM orders")->fetchColumn();
        $txt = "آمار کلی ربات:\n\n"
             . "👥 کاربران: {$users}\n"
             . "🛒 کل سفارش‌ها: {$orders}";
        $telegram->sendMessage($chatId, $txt);
        return true;
    }

    // شارژ همگانی
    if ($text === 'شارژ همگانی 💰') {
        setState($pdo, $user['id'], 'admin_mass_charge');
        $telegram->sendMessage($chatId, "مبلغ شارژ (تومان) را ارسال کنید. این مبلغ به موجودی همه کاربران اضافه می‌شود.");
        return true;
    }

    // ارسال همگانی
    if ($text === 'ارسال همگانی ✉️') {
        setState($pdo, $user['id'], 'admin_mass_broadcast');
        $telegram->sendMessage($chatId, "متن پیام همگانی را ارسال کنید.");
        return true;
    }

    // مدیریت خدمات (راهنما)
    if ($text === 'مدیریت خدمات 📝') {
        $telegram->sendMessage($chatId,
            "برای مدیریت خدمات از دستورات زیر استفاده کنید:\n\n".
            "/add_service - افزودن خدمت جدید\n".
            "/list_services - لیست خدمات\n".
            "/del_service ID - حذف خدمت\n\n".
            "هر خدمت دارای سود عددی و درصدی است."
        );
        return true;
    }

    // مدیریت استارز / پریمیوم / شماره مجازی - فقط راهنمای کوتاه
    if (in_array($text, ['مدیریت استارز ✨','مدیریت پریمیوم 💎','مدیریت شماره مجازی 📱'])) {
        $telegram->sendMessage($chatId, "برای افزودن/ویرایش این نوع خدمات از /add_service استفاده کن و type مناسب را وارد کن (premium/stars/number).");
        return true;
    }

    // مدیریت سفارشات - نمایش ۱۰ سفارش آخر
    if ($text === 'مدیریت سفارشات 🛒') {
        $stmt = $pdo->query("
            SELECT o.id, o.status, o.final_price, u.tg_id, s.title_fa
            FROM orders o
            JOIN users u ON u.id = o.user_id
            JOIN services s ON s.id = o.service_id
            ORDER BY o.id DESC
            LIMIT 10
        ");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$rows) {
            $telegram->sendMessage($chatId, "سفارشی یافت نشد.");
            return true;
        }
        $lines = [];
        foreach ($rows as $r) {
            $lines[] = "#{$r['id']} | {$r['title_fa']} | ".number_format($r['final_price'])." | {$r['status']} | TG: {$r['tg_id']}";
        }
        $telegram->sendMessage($chatId, implode(\"\\n\", $lines));
        return true;
    }

    // جستجو کاربر / سفارش – در این نسخه ساده فقط راهنما
    if ($text === 'جستجو کاربر 🔍') {
        $telegram->sendMessage($chatId, "برای جستجو کاربر، آیدی عددی یا یوزرنیم را با دستور زیر بفرست:\n/search_user VALUE");
        return true;
    }
    if ($text === 'جستجو سفارش 📄') {
        $telegram->sendMessage($chatId, "برای جستجو سفارش، از دستور زیر استفاده کن:\n/search_order ID");
        return true;
    }

    // تنظیمات پرداخت (فقط توضیح کارت به کارت)
    if ($text === 'تنظیمات پرداخت 💳') {
        $telegram->sendMessage($chatId,
            "در حال حاضر فقط پرداخت کارت به کارت فعال است.\n".
            "متن و شماره کارت را می‌توانی در فایل config.php (ثابت CARD_INFO_FA) ویرایش کنی."
        );
        return true;
    }

    // مدیریت دکمه‌ها / وب‌سرویس / کارمندان – پیام راهنما
    if ($text === 'مدیریت دکمه‌ها ⬛') {
        $telegram->sendMessage($chatId, "متن دکمه‌ها در فایل classes/Keyboard.php قابل ویرایش است.");
        return true;
    }
    if ($text === 'تنظیمات وب‌سرویس 🌐') {
        $telegram->sendMessage($chatId, "تنظیمات CALLINO_API_KEY و CALLINO_API_BASE در فایل config.php قرار دارد.");
        return true;
    }
    if ($text === 'مدیریت کارمندان ⌨️') {
        $telegram->sendMessage($chatId, "برای افزودن کارمند جدید، آیدی عددی او را در جدول admins وارد کن یا یک INSERT ساده در دیتابیس بزن.");
        return true;
    }

    // دستورات متنی /approve_X و /reject_X و /add_service و ...
    if (preg_match('~^/approve_(\d+)$~', $text, $m)) {
        adminApprovePayment((int)$m[1], $pdo, $telegram);
        return true;
    }
    if (preg_match('~^/reject_(\d+)$~', $text, $m)) {
        adminRejectPayment((int)$m[1], $pdo, $telegram);
        return true;
    }

    // افزودن خدمت جدید
    if ($text === '/add_service') {
        setState($pdo, $user['id'], 'admin_add_service_step1');
        $telegram->sendMessage($chatId,
            "➕ افزودن خدمت جدید\n\n".
            "مرحله ۱: نوع خدمت را ارسال کنید (یکی از مقادیر زیر):\n".
            "premium\nstars\nnumber"
        );
        return true;
    }

    // ادامه مراحل افزودن خدمت
    [$state, $data] = getState($pdo, $user['id']);

    if ($state === 'admin_add_service_step1') {
        $type = trim(strtolower($text));
        if (!in_array($type, ['premium','stars','number'])) {
            $telegram->sendMessage($chatId, "نوع نامعتبر است. یکی از مقادیر زیر را بفرست:\npremium\nstars\nnumber");
            return true;
        }
        setState($pdo, $user['id'], 'admin_add_service_step2', ['type' => $type]);
        $telegram->sendMessage($chatId, "مرحله ۲: عنوان فارسی خدمت را ارسال کنید.");
        return true;
    }

    if ($state === 'admin_add_service_step2') {
        $data['title_fa'] = $text;
        setState($pdo, $user['id'], 'admin_add_service_step3', $data);
        $telegram->sendMessage($chatId, "مرحله ۳: عنوان انگلیسی خدمت را ارسال کنید.");
        return true;
    }

    if ($state === 'admin_add_service_step3') {
        $data['title_en'] = $text;
        setState($pdo, $user['id'], 'admin_add_service_step4', $data);
        $telegram->sendMessage($chatId, "مرحله ۴: کد سرویس در کالینو (callino_service_code) را ارسال کنید.");
        return true;
    }

    if ($state === 'admin_add_service_step4') {
        $data['callino_service_code'] = $text;
        setState($pdo, $user['id'], 'admin_add_service_step5', $data);
        $telegram->sendMessage($chatId, "مرحله ۵: قیمت پایه (بدون سود) را به تومان به‌صورت عدد بفرست.");
        return true;
    }

    if ($state === 'admin_add_service_step5') {
        if (!ctype_digit($text)) {
            $telegram->sendMessage($chatId, "قیمت باید فقط عدد باشد.");
            return true;
        }
        $data['base_price'] = (int)$text;
        setState($pdo, $user['id'], 'admin_add_service_step6', $data);
        $telegram->sendMessage($chatId, "مرحله ۶: سود عددی (profit_fixed) را به تومان وارد کنید (مثلاً 5000). اگر نمی‌خواهی 0 بفرست.");
        return true;
    }

    if ($state === 'admin_add_service_step6') {
        if (!ctype_digit($text)) {
            $telegram->sendMessage($chatId, "سود عددی باید فقط عدد باشد.");
            return true;
        }
        $data['profit_fixed'] = (int)$text;
        setState($pdo, $user['id'], 'admin_add_service_step7', $data);
        $telegram->sendMessage($chatId, "مرحله ۷: سود درصدی (profit_percent) را ارسال کنید (مثلاً 10). اگر نمی‌خواهی 0 بفرست.");
        return true;
    }

    if ($state === 'admin_add_service_step7') {
        if (!ctype_digit($text)) {
            $telegram->sendMessage($chatId, "سود درصدی باید فقط عدد باشد.");
            return true;
        }
        $data['profit_percent'] = (int)$text;

        $stmt = $pdo->prepare("
            INSERT INTO services (type,title_fa,title_en,callino_service_code,base_price,profit_fixed,profit_percent,active)
            VALUES (?,?,?,?,?,?,?,1)
        ");
        $stmt->execute([
            $data['type'],
            $data['title_fa'],
            $data['title_en'],
            $data['callino_service_code'],
            $data['base_price'],
            $data['profit_fixed'],
            $data['profit_percent'],
        ]);

        clearState($pdo, $user['id']);
        $telegram->sendMessage($chatId, "خدمت جدید با موفقیت ثبت شد ✅");
        return true;
    }

    // لیست خدمات
    if ($text === '/list_services') {
        $stmt = $pdo->query("SELECT id,type,title_fa,base_price,profit_fixed,profit_percent,active FROM services ORDER BY id DESC LIMIT 20");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$rows) {
            $telegram->sendMessage($chatId, "هیچ خدمتی ثبت نشده است.");
            return true;
        }
        $lines = [];
        foreach ($rows as $r) {
            $lines[] = "#{$r['id']} | {$r['type']} | {$r['title_fa']} | قیمت پایه: ".number_format($r['base_price'])." | سود: +".number_format($r['profit_fixed'])." +{$r['profit_percent']}% | ".($r['active'] ? 'فعال' : 'غیرفعال');
        }
        $telegram->sendMessage($chatId, implode(\"\\n\", $lines));
        return true;
    }

    // حذف خدمت
    if (preg_match('~^/del_service\s+(\d+)$~', $text, $m)) {
        $id = (int)$m[1];
        $stmt = $pdo->prepare("DELETE FROM services WHERE id = ?");
        $stmt->execute([$id]);
        $telegram->sendMessage($chatId, "در صورت وجود، خدمت #{$id} حذف شد.");
        return true;
    }

    // شارژ همگانی / ارسال همگانی بر اساس state
    [$state2, $data2] = getState($pdo, $user['id']);

    if ($state2 === 'admin_mass_charge') {
        if (!ctype_digit($text)) {
            $telegram->sendMessage($chatId, "مبلغ باید فقط عدد باشد.");
            return true;
        }
        $amount = (int)$text;
        $pdo->prepare("UPDATE users SET balance = balance + ?")->execute([$amount]);
        clearState($pdo, $user['id']);
        $telegram->sendMessage($chatId, "مبلغ ".number_format($amount)." تومان به موجودی همه کاربران اضافه شد.");
        return true;
    }

    if ($state2 === 'admin_mass_broadcast') {
        $stmt = $pdo->query("SELECT tg_id FROM users");
        $rows = $stmt->fetchAll(PDO::FETCH_COLUMN);
        foreach ($rows as $tgId) {
            $telegram->sendMessage($tgId, $text);
        }
        clearState($pdo, $user['id']);
        $telegram->sendMessage($chatId, "پیام برای ".count($rows)." کاربر ارسال شد.");
        return true;
    }

    // جستجو با دستورات
    if (preg_match('~^/search_user\s+(.+)$~', $text, $m)) {
        $value = trim($m[1]);
        if (ctype_digit($value)) {
            $stmt = $pdo->prepare("SELECT * FROM users WHERE tg_id = ?");
            $stmt->execute([$value]);
        } else {
            $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
            $stmt->execute([$value]);
        }
        $u = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$u) {
            $telegram->sendMessage($chatId, "کاربر یافت نشد.");
            return true;
        }
        $telegram->sendMessage($chatId,
            "کاربر یافت شد:\n".
            "ID: {$u['id']}\nTG: {$u['tg_id']}\nUsername: @{$u['username']}\nBalance: ".number_format($u['balance'])
        );
        return true;
    }

    if (preg_match('~^/search_order\s+(\d+)$~', $text, $m)) {
        $id = (int)$m[1];
        $stmt = $pdo->prepare("
            SELECT o.*, u.tg_id, s.title_fa
            FROM orders o
            JOIN users u ON u.id = o.user_id
            JOIN services s ON s.id = o.service_id
            WHERE o.id = ?
        ");
        $stmt->execute([$id]);
        $o = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$o) {
            $telegram->sendMessage($chatId, "سفارشی با این شناسه یافت نشد.");
            return true;
        }
        $telegram->sendMessage($chatId,
            "سفارش #{$o['id']}:\n".
            "کاربر: {$o['tg_id']}\n".
            "خدمت: {$o['title_fa']}\n".
            "هدف: {$o['target']}\n".
            "مبلغ: ".number_format($o['final_price'])."\n".
            "تعداد: {$o['quantity']}\n".
            "وضعیت: {$o['status']}"
        );
        return true;
    }

    return false;
}

function adminApprovePayment($paymentId, PDO $pdo, Telegram $telegram) {
    $stmt = $pdo->prepare("
        SELECT p.*, u.tg_id, o.service_id, o.target, o.final_price, o.quantity
        FROM payments p
        JOIN users u ON u.id = p.user_id
        JOIN orders o ON o.id = p.order_id
        WHERE p.id = ?
    ");
    $stmt->execute([$paymentId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        $telegram->sendMessage(OWNER_ID, "پرداخت یافت نشد.");
        return;
    }

    if ($row['status'] !== 'pending') {
        $telegram->sendMessage(OWNER_ID, "این پرداخت قبلاً بررسی شده است.");
        return;
    }

    $pdo->beginTransaction();
    $pdo->prepare("UPDATE payments SET status = 'approved' WHERE id = ?")->execute([$paymentId]);
    $pdo->prepare("UPDATE orders SET status = 'paid' WHERE id = ?")->execute([$row['order_id']]);
    $pdo->commit();

    // اطلاع به کاربر
    $telegram->sendMessage($row['tg_id'],
        "پرداخت شما برای سفارش #{$row['order_id']} تأیید شد ✅\n".
        "سفارش در حال انجام است."
    );

    $telegram->sendMessage(OWNER_ID, "پرداخت #{$paymentId} تأیید شد.");

    // در اینجا می‌توانی اتصال به کالینو را برای انجام خودکار سفارش اضافه کنی.
    // با توجه به نوع سرویس، از CallinoApi استفاده کن.
}

function adminRejectPayment($paymentId, PDO $pdo, Telegram $telegram) {
    $stmt = $pdo->prepare("
        SELECT p.*, u.tg_id, o.id AS oid
        FROM payments p
        JOIN users u ON u.id = p.user_id
        JOIN orders o ON o.id = p.order_id
        WHERE p.id = ?
    ");
    $stmt->execute([$paymentId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        $telegram->sendMessage(OWNER_ID, "پرداخت یافت نشد.");
        return;
    }

    $pdo->beginTransaction();
    $pdo->prepare("UPDATE payments SET status = 'rejected' WHERE id = ?")->execute([$paymentId]);
    $pdo->prepare("UPDATE orders SET status = 'canceled' WHERE id = ?")->execute([$row['oid']]);
    $pdo->commit();

    $telegram->sendMessage($row['tg_id'],
        "❌ پرداخت شما برای سفارش #{$row['oid']} رد شد.\n".
        "در صورت نیاز با پشتیبانی تماس بگیرید."
    );
    $telegram->sendMessage(OWNER_ID, "پرداخت #{$paymentId} رد شد.");
}
